<?php
/**
 *  Main configuration
 *
 * @category PHP
 * @package  VenoFileManager
 * @author   Nicola Franchini <info@veno.it>
 * @license  Exclusively sold on CodeCanyon
 * @link     http://filemanager.veno.it/
 */
if (!class_exists('SetUp', false)) {
    /**
     * SetUp Class
     *
     * @category PHP
     * @package  VenoFileManager
     * @author   Nicola Franchini <info@veno.it>
     * @license  Exclusively sold on CodeCanyon
     * @link     http://filemanager.veno.it/
     */
    class SetUp
    {
        public $lang;
        /**
        * Set session and language
        *
        * @param string $relative relative path to call
        *
        * @return set lang session
        */
        public function __construct($relative = 'vfm-admin/')
        {
            $timeconfig = $this->getConfig('default_timezone');
            $timezone = (strlen($timeconfig) > 0) ? $timeconfig : "UTC";
            date_default_timezone_set($timezone);

            if (!isset($_SESSION)) {
                if (strlen($this->getConfig('session_name')) > 0) {
                    session_name($this->getConfig('session_name'));
                }
                session_start();
            }

            $this->lang = $this->getConfig('lang');

            if (isset($_GET['lang'])
                && file_exists($relative.'translations/'.$_GET['lang'].'.php')
            ) {
                $this->lang = $_GET['lang'];
                $_SESSION['lang'] = $_GET['lang'];
            }
            if (isset($_SESSION['lang'])) {
                $this->lang = $_SESSION['lang'];
            }
        }

        /**
        * Get string in current language
        *
        * @param string $stringName string to translate
        *
        * @return translated string
        */
        public function getString($stringName)
        {
            return $this->getLangString($stringName, $this->lang);
        }

        /**
        * The function for getting a translated string.
        * Falls back to english if the correct language is missing something.
        *
        * @param string $stringName string to translate
        *
        * @return translation
        */
        public static function getLangString($stringName)
        {
            global $_TRANSLATIONS;
            if (isset($_TRANSLATIONS)
                && is_array($_TRANSLATIONS)
                && isset($_TRANSLATIONS[$stringName])
                && strlen($_TRANSLATIONS[$stringName]) > 0
            ) {
                $translation = htmlspecialchars($_TRANSLATIONS[$stringName], ENT_COMPAT, 'UTF-8', false);

                // Don't bother if there are no specialchars - saves some processing.
                if (!preg_match('/[&<>"\']/', $translation)) {
                    return $translation;
                }
                $translation = stripslashes(strip_tags($translation));
                return $translation;
            } else {
                return '&gt;'.stripslashes(strip_tags(htmlspecialchars($stringName))).'&lt;';
            }
        }

        /**
        * Print languages list
        *
        * @param string $dir realtive path to translations
        *
        * @return Languages list
        */
        public function printLangMenu($dir = '')
        {
            global $translations_index;
            $menu = '';
            $files = glob($dir.'translations/*.php');
            parse_str($_SERVER['QUERY_STRING'], $params);
            unset($params['lang']);
            $query = http_build_query($params);
            $query = $query !== '' ? '?'.$query.'&' : '?';

            foreach ($files as $item) {
                $fileinfo = Utils::mbPathinfo($item);
                $langvar = $fileinfo['filename'];
                $menu .= '<li';
                if ($this->lang === $langvar) {
                    $menu .= ' class="active"';
                }
                $menu .='><a href="'.$query.'lang='.$langvar.'">';
                $out = isset($translations_index[$langvar]) ? $translations_index[$langvar] : $langvar;
                $menu .= '<span>'.$out.'</span></a></li>';
            }
            return $menu;
        }

        /**
        * The function outputting the shortcut icon tag
        *
        * @param string $path relative path to the file
        *
        * @return $output
        */
        public function printIcon($path = '')
        {
            $icon_ico = $path.'favicon.ico';
            $icon_png = $path.'favicon-152.png';
            $output = '';

            if (file_exists($icon_ico)) {
                $output .= '<link rel="shortcut icon" href="'.$icon_ico.'">';
            }
            if (file_exists($icon_png)) {
                $output .= '<link rel="apple-touch-icon-precomposed" href="'.$icon_png.'">';
            }
            return $output;
        }

        /**
        * The function for getting configuration values
        *
        * @param string $name    config option name
        * @param string $default optional default value
        *
        * @return config value
        */
        public static function getConfig($name, $default = false)
        {
            global $_CONFIG;
            if (isset($_CONFIG) && isset($_CONFIG[$name])) {
                if ($_CONFIG[$name] !== false) {
                    return $_CONFIG[$name];
                }
            }
            return $default;
        }

        /**
        * Get app description
        *
        * @return html decoded description or false
        */
        public function getDescription()
        {
            $fulldesc = html_entity_decode(Setup::getConfig('description'), ENT_QUOTES, 'UTF-8');
            $cleandesc = strip_tags($fulldesc, '<img>');

            if (strlen(trim($cleandesc)) > 0) {
                return $fulldesc;
            }
            return false;
        }

        /**
        * Print alert messages
        *
        * @return the alert
        */
        public function printAlert()
        {
            $_ERROR = isset($_SESSION['error']) ? $_SESSION['error'] : false;
            $_SUCCESS = isset($_SESSION['success']) ? $_SESSION['success'] : false;
            $_WARNING = isset($_SESSION['warning']) ? $_SESSION['warning'] : false;

            $alert = false;
            $output = '';
            $sticky_class = '';

            if (SetUp::getConfig('sticky_alerts') === true) {
                $sticky_class = 'sticky-alert '.SetUp::getConfig('sticky_alerts_pos');
            }

            $closebutt = '<button type="button" class="close" aria-label="Close"><span aria-hidden="true">&times;</span></button>';

            if ($_ERROR) {
                $alert = true;
                $output .= '<div class="response nope alert" role="alert">'.$_ERROR.$closebutt.'</div>';
            }
            if ($_WARNING) {
                $alert = true;
                $output .= '<div class="response boh alert" role="alert">'.$_WARNING.$closebutt.'</div>';
            }
            if ($_SUCCESS) {
                $alert = true;
                $output .= '<div class="response yep alert" role="alert">'.$_SUCCESS.$closebutt.'</div>';
            }
            if ($alert === true) {
                $output = '<div class="alert-wrap '.$sticky_class.'">'.$output.'</div>';
                unset($_SESSION['success']);
                unset($_SESSION['error']);
                unset($_SESSION['warning']);
                return $output;
            }
            return false;
        }

        /**
        * Format modification date time
        *
        * @param string $time new format
        *
        * @return formatted date
        */
        public static function formatModTime($time)
        {
            $timeformat = 'd.m.y H:i:s';
            if (SetUp::getConfig('time_format') != null
                && strlen(SetUp::getConfig('time_format')) > 0
            ) {
                $timeformat = SetUp::getConfig('time_format');
            }
            return date($timeformat, $time);
        }

        /**
        * Format file size
        *
        * @param string $size new format
        *
        * @return formatted size
        */
        public function formatSize($size)
        {
            $sizes = array('B', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB');
            $syz = $sizes[0];
            for ($i = 1; (($i < count($sizes)) && ($size >= 1024)); $i++) {
                $size = $size / 1024;
                $syz  = $sizes[$i];
            }
            return round($size, 2).' '.$syz;
        }

        /**
        * Get all users from users.php
        *
        * @return users array
        */
        public static function getUsers()
        {
            global $_USERS;
            if (isset($_USERS)) {
                return $_USERS;
            }
            return false;
        }

        /**
        * Return ini values in bytes
        *
        * @param string $val value to convert
        *
        * @return value in bytes
        */
        public function returnIniBytes($val)
        {
            $val = trim($val);
            $last = strtolower($val[strlen($val)-1]);

            $val = floatval($val);

            switch($last) 
            {
            case 'g':
                $val *= 1024;
            case 'm':
                $val *= 1024;
            case 'k':
                $val *= 1024;
            }
            return $val;
        }

        /**
        * Server available max upload size
        *
        * @return available max upload size in bytes
        */
        public function getMaxUploadSize()
        {
            // select maximum upload size
            $max_upload = ($this->returnIniBytes(ini_get('upload_max_filesize'))/2);
            // select post limit
            $max_post = ($this->returnIniBytes(ini_get('post_max_size'))/2);
            // select memory limit
            $init_memory_limit = ini_get('memory_limit');
            // set equal to post_max_size if memory_limit is unlimited
            $memory_limit = $init_memory_limit == '-1' ? $max_post : ($this->returnIniBytes($init_memory_limit)/2);
            // get the smallest of them, this defines the real limit
            $available = min($max_upload, $max_post, $memory_limit);
            // Set 1MB as minimum safe
            $available_safe = max(1048576, $available);
            // return the value in bytes
            return round($available_safe);
        }

        /**
        * Upload chunk size
        *
        * @return chunksize in bytes (32M if the server can upload at least 64M).
        */
        public function getChunkSize()
        {
            $serverSize = $this->getMaxUploadSize();
            $idealSize = 32*1048576; // 1048576 == 1MB
            return min($serverSize, $idealSize);
        }
    }
}
